/**
* \file: IAudioOutAdapter.h
*
* \version: $Id:$
*
* \release: $Name:$
*
* <brief description>.
* <detailed description>
* \component: CarPlay
*
* \author: J. Harder / ADIT/SW1 / jharder@de.adit-jv.com
*
* \copyright (c) 2013-2014 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
* \see <related items>
*
* \history
*
***********************************************************************/

#ifndef CARPLAY_IAUDIOOUTADAPTER_H
#define CARPLAY_IAUDIOOUTADAPTER_H

#include "IConfiguration.h"

namespace adit { namespace carplay
{

/**
 * Audio format as used in configuration.
 */
enum AudioFormat
{
    AudioFormat_Invalid            = 0x00000000,
    AudioFormat_8KHz_16Bit_Mono    = 0x00000001,
    AudioFormat_8KHz_16Bit_Stereo  = 0x00000002,
    AudioFormat_16KHz_16Bit_Mono   = 0x00000004,
    AudioFormat_16KHz_16Bit_Stereo = 0x00000008,
    AudioFormat_24KHz_16Bit_Mono   = 0x00000010,
    AudioFormat_24KHz_16Bit_Stereo = 0x00000020,
    AudioFormat_32KHz_16Bit_Mono   = 0x00000040,
    AudioFormat_32KHz_16Bit_Stereo = 0x00000080,
    AudioFormat_44KHz_16Bit_Mono   = 0x00000100,
    AudioFormat_44KHz_16Bit_Stereo = 0x00000200,
    AudioFormat_44KHz_24Bit_Mono   = 0x00000400,
    AudioFormat_44KHz_24Bit_Stereo = 0x00000800,
    AudioFormat_48KHz_16Bit_Mono   = 0x00001000,
    AudioFormat_48KHz_16Bit_Stereo = 0x00002000,
    AudioFormat_48KHz_24Bit_Mono   = 0x00004000,
    AudioFormat_48KHz_24Bit_Stereo = 0x00008000
} __attribute__ ((packed, aligned (8)));

typedef int64_t AudioFormats;

struct AudioFormatStruct
{
    uint16_t SampleRate;
    uint8_t BitsPerChannel;
    uint8_t Channels;
    double vocoderSampleRate;

    AudioFormatStruct();
    AudioFormatStruct(AudioFormat format);
    AudioFormat ToEnum();
} __attribute__ ((packed, aligned (4)));

struct Samples
{
    void*       DataPtr;
    size_t      Length;
    uint64_t    TimeStamp;
};

/**
 * Audio channel type
 * Main:      music, speech, telephony, etc.
 * Alternate: UI beeps, turn-by-turn, etc.
 */
typedef enum
{
    AudioChannelType_Main = 0,
    AudioChannelType_Alternate = 1,
    AudioChannelType_Main_High = 2
} __attribute__((packed, aligned (1))) AudioChannelType;


typedef enum
{
    AudioStreamType_Compatibility = 0,
    AudioStreamType_Default = 1,
    AudioStreamType_Media = 2,
    AudioStreamType_Telephony = 3,
    AudioStreamType_SpeechRecognition = 4,
    AudioStreamType_Alert = 5,
} __attribute__((packed, aligned (1))) AudioType;


/**
 * Interface provided by CarPlay.
 */
class IAudioOutSource
{
public:
    /**
     * Read samples from CarPlay.
     *
     * \param [in] samples      buffer to write the audio data
     */
    virtual void Read(const Samples& samples) = 0;

protected:
    /**
     * Default virtual destructor.
     */
    virtual ~IAudioOutSource() {};
};

/**
 * Interface required for audio out play back.
 *
 * Thread-safety is guaranteed by CarPlay.
 */
class IAudioOutAdapter
{
public:
    /**
     * Default virtual descructor.
     */
    virtual ~IAudioOutAdapter() {};

    /**
     * Initializes audio pipeline.
     *
     * \param [in] config   configuration
     * \param [in] source   source to read audio from
     *
     * \return bool         false: if failed to initialize pipeline
     */
    virtual bool Initialize(const IConfiguration& config, IAudioOutSource& source) = 0;

    /**
     * Starts audio pipeline.
     *
     * \param [in] format   kHz, Bit and Stereo/Mono settings
     * \param [in] channel  Main or alternate audio channel
     * \param [in] audioType     type of audio
     *
     * \return bool         false: if failed to start pipeline
     *
     * \see Apple specification for audio types.
     */
    virtual bool Prepare(AudioFormatStruct format, AudioChannelType channel,
            const std::string& audioType) = 0;

    /**
     * Set audio pipeline to playing.
     *
     * \return bool         false: if failed to play pipeline
     */
    virtual bool Start() = 0;

    /**
     * Stops audio pipeline.
     */
    virtual void Stop() = 0;

    /**
     * Flush Audio pipeline.
     */
    virtual void Flush() = 0;
};

} } /* namespace adit { namespace carplay */

#endif /* CARPLAY_IAUDIOOUTADAPTER_H */
